/*
 * cProtocolStack.cpp
 *
 * The actual protocol stack.
 */
#include "cProtocolStack.h"
#include "Interfaces/cLayer.h"
#include "Interfaces/cCallback.h"
#include "cGroup.h"
#include "cMsgBuffer.h"
#include "cBufferManager.h"
#include "cLayerInfo.h"

#include "Util/cException.h"
#include "Util/gError.h"

#define DEFAULT_BUFFERMGR_SIZE		20
#define MAX_EVENTS					16
#define SCHEDULE_INTERVAL			300	// Schedule 3 times a second
#define DEFAULT_ADDRESS_GROUP_SIZE	8

bool  cProtocolStack::mSimulate    = false;
DWORD cProtocolStack::mCurrentTime = 0;

/*
 * cProtocolStack::TimeStep()
 */
void cProtocolStack::TimeStep(DWORD stepVal)
{
	mSimulate = true;
	mCurrentTime += stepVal;
}


/*
 * cProtocolStack::cProtocolStack()
 *
 * Purpose:	Create a new protocol stack.
 * IN:		-	
 */
cProtocolStack::cProtocolStack(char* setupString)
{
	mRunMain    = false;
	mLayerBelow = NULL;		// No bottom layer yet.
	mEventHandleArray = NULL;
	mCallbackArray	  = NULL;
	mLocalAddressGroup = NULL;
	mNumEvents  = 0;
	mNextSchedule = 0;

	// Create the local address group
	mLocalAddressGroup = new cGroup(DEFAULT_ADDRESS_GROUP_SIZE);
	if(!mLocalAddressGroup)
	{
		gError("Unable to create local address group.", __LINE__, __FILE__);
		throw cException("<cProtocolStack::cProtocolStack>: Unable to create local address group.");
	}

	// Create event handling array
	mEventHandleArray	= new HANDLE[MAX_EVENTS];
	mCallbackArray		= new cCallback*[MAX_EVENTS];
	mCallbackParamArray = new void*[MAX_EVENTS];
	if(!mEventHandleArray || !mCallbackArray || !mCallbackParamArray)
	{
		gError("Unable to create event arrays.", __LINE__, __FILE__);
		throw cException("<cProtocolStack::cProtocolStack>: Unable to create event arrays.");
	}

	// Create an event for when the thread should stop.
	mThreadStopEvent = CreateEvent(NULL,	// Same security attribs as parent.
								   FALSE,	// Automatic reset.
								   FALSE,	// Set to not-signaled state
								   NULL);
	if(!mThreadStopEvent)
	{
		gError("Unable to create thread stop event.", __LINE__, __FILE__);
		throw cException("<cProtocolStack::cProtocolStack>: Unable to create thread stop event.");
	}
	else
	{
		mEventHandleArray[0]	= mThreadStopEvent;
		mCallbackArray[0]		= NULL;
		mCallbackParamArray[0]	= NULL;
		mNumEvents = 1;
	}
}

/*
 * cProtocolStack::~cProtocolStack()
 *
 * Purpose:	Destructor
 * IN:		-	
 */
cProtocolStack::~cProtocolStack()
{
	Cleanup();
	CloseHandle(mThreadStopEvent);
	if(mEventHandleArray) { delete mEventHandleArray; }
	if(mCallbackArray)	  { delete mCallbackArray;	  }
	if(mLocalAddressGroup){ delete mLocalAddressGroup;}
}

/*
 * cProtocolStack::AddEvent()
 *
 * Purpose:	Adds an event to the event processing loop.
 * IN:		event		-> The object to add to the event list.
 *			callback	-> The callback object.
 *			param		-> The parameter to pass to the callback.
 * OUT:		-
 * Cond:	There must be room in the event table.
 * PostCnd:	The event is added and monitored...callback called when it occurs.
 * Return:	true if success, else false
 */
bool cProtocolStack::AddEvent(HANDLE event, cCallback* callback, void* param)
{

	mCS.Enter();

	if(mNumEvents == MAX_EVENTS)
	{
		gError("Too many events!", __LINE__, __FILE__);
		mCS.Leave();
		return false;
	}

	// Note: use mNumEvents because array starts at 0!
	mEventHandleArray[mNumEvents]	= event;
	mCallbackArray[mNumEvents]		= callback;
	mCallbackParamArray[mNumEvents] = param;
	mNumEvents++;

	mCS.Leave();
	return true;
}

/*
 * cProtocolStack::RemoveEvent()
 *
 * Purpose:	Removes a previously registered event.
 * IN:		event		-> The event to remove.
 * OUT:		-
 * Cond:	The event must be in the event table.
 * PostCnd:	The event is removed.
 * Return:	true if success, else false
 */
bool cProtocolStack::RemoveEvent(HANDLE event)
{
	mCS.Enter();

	// Remove event

	mCS.Leave();
	return false;
}

/*
 * cProtocolStack::Cleanup()
 *
 * Purpose:	Cleans up all of the layers of the stack.
 * IN:		-
 * OUT:		-
 * Cond:	Stack was set up.
 * PostCnd:	Stack is destroyed and useless.
 * Return:	true if success, else false
 */
bool cProtocolStack::Cleanup()
{
	mCS.Enter();

	if(mRunMain)
	{
		_StopMain();
	}

	cLayerInfo* info;
	while(mLayerBelow != NULL)
	{
		info = mLayerBelow->GetLayerInfo();
		mLayerBelow->Cleanup();
		mLayerBelow = info->GetLayerBelow();
	}

	mCS.Leave();
	return true;
}

/*
 * cProtocolStack::AllocateBuffer()
 *
 * Purpose:	Gets a message buffer for use.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	A message buffer is pulled off the free list.
 * Return:	NULL if fail, else a valid pointer to a message buffer.
 */
cMsgBuffer*	cProtocolStack::AllocateBuffer(int size)
{
	cMsgBuffer* buffer = cBufferManager::GetBuffer();
	if(cMsgBuffer)
	{
		if(!buffer->Init(mCumHeaderSize+size, mCumHeaderSize))
		{
			buffer->Release();
			buffer = NULL;
		}
	}
	return buffer;
}

/*
 * cProtocolStack::Send()
 *
 * Purpose:	Sends a message down through the stack (single copy)
 * IN:		dest		-> The destination for the message.
 *			buffer		-> Char buffer that is the message.
 *			size		-> The size of the buffer.
 * OUT:		-
 * Cond:	-
 * PostCnd:	A message buffer is pulled off the free list and used to encapsulate payload.
 * Return:	true if success, else false
 */
bool cProtocolStack::Send(cGroup* dest, char* buffer, int size, int messageType)
{
	cMsgBuffer	*buf;

	mCS.Enter();

	if(mLayerBelow)
	{
		// Send here.
		buf = AllocateBuffer(size);
		buf->SetPayload(buffer, size);

		buf->AddRef();
		mLayerBelow->Send(dest, buf, messageType);
		buf->Release();
	}	

	mCS.Leave();
	return false;
}

/*
 * cProtocolStack::Send()
 *
 * Purpose:	Sends a message buffer down through the stack (zero copy)
 * IN:		dest		-> The destination for the message.
 *			buffer		-> The message to be sent (buffer must have been from this stack!)
 * OUT:		-
 * Cond:	-
 * PostCnd:	The buffer is sent down the stack.
 * Return:	true if success, else false
 */
bool cProtocolStack::Send(cGroup* dest, cMsgBuffer* buffer, int messageType)
{

	mCS.Enter();

	if(mLayerBelow)
	{
		buffer->AddRef();
		mLayerBelow->Send(dest, buffer, messageType);
		buffer->Release();
	}

	mCS.Leave();
	return true;
}

/*
 * cProtocolStack::RegisterDeliverCallback()
 *
 * Purpose:	Registers a callback for message delivery.
 * IN:		callback	-> Object that implements Deliver callback interface.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The callback is added to msg receipt callback list.
 * Return:	NULL if fail, else cLayerInfo to be used to get rid of callback.
 */
bool cProtocolStack::RegisterDeliverCallback(cHandle* handle, cDeliver* callback)
{
	bool	retVal = false;
	mCS.Enter();
		if(mRunMain)
		{
			retVal = false;
		}
		else if(mLayerBelow)
		{
			retVal = mLayerBelow->RegisterDeliverCallback(handle, callback);
		}
	mCS.Leave();
	return retVal;
}

/*
 * cProtocolStack::UnregisterDeliverCallback()
 *
 * Purpose:	Unregisters a previously registered callback.
 * IN:		handle		-> The handle received on register.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The callback is no longer called on msg receipt.
 * Return:	true if success, else false
 */
bool cProtocolStack::UnregisterDeliverCallback(cHandle handle)
{
	bool	retVal = false;
	mCS.Enter();
		if(mRunMain)
		{
			retVal = false;
		}
		else if(mLayerBelow)
		{
			retVal = mLayerBelow->UnregisterDeliverCallback(handle);
		}
	mCS.Leave();
	return retVal;
}


unsigned int cProtocolStack::GetMTS()
{
	if(mLayerBelow)
	{
		return mLayerBelow->GetLayerInfo()->GetMTS();
	}
	else
	{
		return 0;
	}
}

/*
 * cProtocolStack::GetLocalAddress()
 *
 * Purpose:	Gets the local endpoint object.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	The callback is no longer called on msg receipt.
 * Return:	NULL if fail, else the local endpoint object.
 */
cGroup*	cProtocolStack::GetLocalAddress()
{
	return mLocalAddressGroup;
}


/*
 * cProtocolStack::AddLocalAddress()
 *
 * Purpose:	Adds an endpoint to the local address group.
 * IN:		endpoint		-> The endpoint to add.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The address is added to the local address list.
 * Return:	true if success, else false.
 */
bool cProtocolStack::AddLocalAddress(cEndpoint* endpoint)
{
	return mLocalAddressGroup->AddEndpoint(endpoint);
}

/*
 * cProtocolStack::AddLayer()
 *
 * Purpose:	Adds a layer to the current stack.
 * IN:		newLayer	-> The new layer to add to the stack
 *			param		-> The parameter set for this layer.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The stack is increased by one layer.
 * Return:	true if success, else false
 */
bool cProtocolStack::AddLayer(cLayer* newLayer, cParam* param)
{

	mCS.Enter();

	// Sanity check
	if(mRunMain || !newLayer || !param)
	{
		mCS.Leave();
		return false;
	}

	// Initialize the new layer.
	param->mProtocolStack = this;
	if(!newLayer->Init(mLayerBelow, param))
	{
		mCS.Leave();
		return false;
	}

	// Update the topmost layer (layer just below the top of stack)
	mLayerBelow = newLayer;
	mCumHeaderSize = mLayerBelow->GetLayerInfo()->GetCumHeaderSize();

	mCS.Leave();
	return true;
}

/*
 * cProtocolStack::RemoveLayer()
 *
 * Purpose:	Removes the top layer from the stack.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	The stack is decreased by one layer; cleanup is called on the layer being removed.
 * Return:	The layer removed if success, else NULL if there are no layers.
 */
cLayer* cProtocolStack::RemoveLayer()
{
	cLayer* retLayer;

	mCS.Enter();

	if(mRunMain)
	{
		mCS.Leave();
		return NULL;
	}

	retLayer = mLayerBelow;
	if(retLayer)
	{
		mLayerBelow = retLayer->GetLayerInfo()->GetLayerBelow();
		retLayer->Cleanup();
	}

	mCS.Leave();
	return retLayer;
}

/*
 * cProtocolStack::Start()
 *
 * Purpose:	Starts the protocol stack executing.
 * IN:		-
 * OUT:		-
 * Cond:	The stack is set up.
 * PostCnd:	The stack can no longer be tampered with; additional thread is run.
 * Return:	true if success, else false
 */
bool cProtocolStack::Start()
{
	mCS.Enter();

	if(mRunMain)
	{
		mCS.Leave();
		return false;
	}

	// Thread settings vars
	LPSECURITY_ATTRIBUTES	lpThreadAttributes;
	DWORD					dwStackSize;
	LPTHREAD_START_ROUTINE	lpStartAddress;
	LPVOID					lpParameter;
	DWORD					dwCreationFlags;
	LPDWORD					lpThreadId;

	// Allow threads to run
	mRunMain = true;

	// Set up the main thread settings.
	lpThreadId			= &mMainThreadId;
	dwCreationFlags		= NULL;
	lpParameter			= this;	//Address of this object.		
	dwStackSize			= 0;	//Same as starting parent.
	lpStartAddress		= cProtocolStack::_ProtocolThread;
	lpThreadAttributes	= NULL;

	// Start the main thread.
	mMainThreadHandle = CreateThread(
		lpThreadAttributes,  // pointer to security attributes
		dwStackSize,         // initial thread stack size
		lpStartAddress,      // pointer to thread function
		lpParameter,         // argument for new thread
		dwCreationFlags,     // creation flags
		lpThreadId           // pointer to receive thread ID
	);
	if(mMainThreadHandle == NULL)
	{
		mRunMain = false;
	}
	
	mCS.Leave();
	return mRunMain;
}

/*
 * cProtocolStack::Stop()
 *
 * Purpose:	Stops the protocol stack from executing.
 * IN:		-
 * OUT:		-
 * Cond:	The stack is running.
 * PostCnd:	The thread is no longer running.  The stack can be re-arranged and restarted.
 * Return:	true if success, else false
 */
bool cProtocolStack::Stop()
{
	bool retVal;

	mCS.Enter();
	if(!mRunMain)
	{
		retVal = false;
	}
	else 
	{ 
		retVal =  _StopMain();
	}
	
	mCS.Leave();
	return retVal;
}

/*
 * cProtocolStack::Schedule()
 *
 * Purpose:	Used for manual scheduling of the protocol stack.
 *			**Recommended only for advanced users**
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	
 * Return:	The exit thread return value.
 */
bool cProtocolStack::Schedule()
{
	DWORD	retVal;
	DWORD	currentTime;
	int		index;
//	mRunMain = true;

	currentTime = cProtocolStack::GetTime();
	if(currentTime > mNextSchedule)
	{
		mNextSchedule = currentTime + SCHEDULE_INTERVAL;	// Set up next schedule time.

		// Schedule all of the layers.
		cLayerInfo* info;
		cLayer*	   below;
		below = mLayerBelow;
		while(below != NULL)
		{
			info = below->GetLayerInfo();
			below->Schedule();
			below = info->GetLayerBelow();
		}
	}

	retVal = WaitForMultipleObjects(mNumEvents, mEventHandleArray, FALSE, 0);
	if(retVal != WAIT_TIMEOUT)
	{
		if((retVal >= WAIT_ABANDONED_0) && (retVal < mNumEvents))
		{
			// Event was abandoned????
			gAbort("An event was abandoned.", __LINE__, __FILE__);
		}
		else
		{
			index = retVal - WAIT_OBJECT_0;
			if(mCallbackArray[index] != NULL)
			{
				mCallbackArray[index]->Callback(mEventHandleArray[index], mCallbackParamArray[index]);
			}
		}
	}
	return true;
}

/*
 * cProtocolStack::_Main()
 *
 * Purpose:	The main for the protocol thread.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	The main executes.
 * Return:	The exit thread return value.
 */
DWORD	cProtocolStack::_Main()
{
	DWORD	retVal;
	DWORD	waitTime;
	DWORD	nextSchedule;
	DWORD	currentTime;
	int		index;

	nextSchedule = cProtocolStack::GetTime() + SCHEDULE_INTERVAL;	
	while(mRunMain)		// mRunMain)
	{
		currentTime = cProtocolStack::GetTime();
		retVal = WAIT_TIMEOUT;						// Prepare to schedule immediately if necessary.
		if(currentTime < nextSchedule)
		{
			waitTime = nextSchedule - currentTime;	// See how long until next schedule time.
			// Wait on event queue.
			retVal = WaitForMultipleObjects(mNumEvents, mEventHandleArray, FALSE, waitTime);
		}
		if(retVal == WAIT_TIMEOUT)
		{
			nextSchedule = cProtocolStack::GetTime() + SCHEDULE_INTERVAL;	// Set up next schedule time.

			// Schedule all of the layers.
			cLayerInfo* info;
			cLayer*	   below;
			below = mLayerBelow;
			while(below != NULL)
			{
				info = below->GetLayerInfo();
				below->Schedule();
				below = info->GetLayerBelow();
			}
		}
		else	// Not a timeout.
		{
			if((retVal >= WAIT_ABANDONED_0) && (retVal < mNumEvents))
			{
				// Event was abandoned????
				gAbort("An event was abandoned.", __LINE__, __FILE__);
			}
			else
			{
				index = retVal - WAIT_OBJECT_0;
				if(mCallbackArray[index] != NULL)
				{
					mCallbackArray[index]->Callback(mEventHandleArray[index], mCallbackParamArray[index]);
				}
			}
		}
	}
	return 1;
}

/*
 * cProtocolStack::_StopMain()
 *
 * Purpose:	Shuts down the main thread.
 * IN:		-
 * OUT:		-
 * Cond:	The main thread is running.
 * PostCnd:	The main thread stops.
 * Return:	true if success, else false
 */
bool cProtocolStack::_StopMain()
{
	DWORD	retCode = STILL_ACTIVE;

	mRunMain = false;
	SetEvent(mThreadStopEvent);	// Wake up the thread if it is sleeping.

	while(retCode == STILL_ACTIVE)
	{
		Sleep(3);
		GetExitCodeThread(mMainThreadHandle, &retCode);
	}
	gInfo("Stop(): Main Thread done.", __LINE__, __FILE__);
	return true;
}

/*
 * cProtocolStack::_ProtocolThread()
 *
 * Purpose:	Starts up execution of the protocol thread.
 * IN:		-
 * OUT:		-
 * Cond:	The main thread is running.
 * PostCnd:	The main thread stops.
 * Return:	true if success, else false
 */
DWORD WINAPI cProtocolStack::_ProtocolThread(LPVOID lpParameter)
{
	DWORD		 retCode = 0;
	cProtocolStack		*stack;
	gInfo("Inside of thread...whoopee!", __LINE__, __FILE__);
	
	//Start up the bundle
	stack = (cProtocolStack*)lpParameter;
	retCode = stack->_Main();

	//Shut down the thread.
	gInfo("Shutting down protocol thread.", __LINE__, __FILE__);
	ExitThread(retCode);

	//To avoid warning
	return retCode;
}

bool cProtocolStack::RegisterViewCallback(cHandle* handle, cView* callback)
{
	bool retVal = false;

	mCS.Enter();
	if(mRunMain)
	{
		retVal = false;
	}
	else if(mLayerBelow)
	{
		retVal = mLayerBelow->RegisterViewCallback(handle, callback);
	}
	mCS.Leave();
	return retVal;
}
bool cProtocolStack::UnregisterViewCallback(cHandle handle)
{
	bool retVal;

	mCS.Enter();
	if(mRunMain)
	{
		retVal = false;
	}
	else if(mLayerBelow)
	{
		retVal = mLayerBelow->UnregisterViewCallback(handle);
	}
	mCS.Leave();
	return retVal;
}

bool cProtocolStack::RegisterErrorCallback(cHandle* handle, cErrorCallback* callback)
{
	bool retVal = false;

	mCS.Enter();
	if(mRunMain)
	{
		retVal = false;
	}
	else if(mLayerBelow)
	{
		retVal = mLayerBelow->RegisterErrorCallback(handle, callback);
	}
	mCS.Leave();
	return retVal;

}

bool cProtocolStack::UnregisterErrorCallback(cHandle handle)
{
	bool retVal;

	mCS.Enter();
	if(mRunMain)
	{
		retVal = false;
	}
	else if(mLayerBelow)
	{
		retVal = mLayerBelow->UnregisterErrorCallback(handle);
	}
	mCS.Leave();
	return retVal;
}